<?php
require_once 'config.php';

if(!isLoggedIn()) {
    header("Location: login.php");
    exit();
}

$user_id = $_SESSION['user_id'];
$error = '';
$success = '';

// Get user data
$stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$user_result = $stmt->get_result();
$user = $user_result->fetch_assoc();

// Update profile
if($_SERVER['REQUEST_METHOD'] == 'POST') {
    if(isset($_POST['update_profile'])) {
        $telegram = sanitize($_POST['telegram_username']);
        $whatsapp = sanitize($_POST['whatsapp_number']);
        
        $update_stmt = $conn->prepare("UPDATE users SET telegram_username = ?, whatsapp_number = ? WHERE id = ?");
        $update_stmt->bind_param("ssi", $telegram, $whatsapp, $user_id);
        
        if($update_stmt->execute()) {
            $success = "Profile updated successfully!";
            $user['telegram_username'] = $telegram;
            $user['whatsapp_number'] = $whatsapp;
        } else {
            $error = "Failed to update profile";
        }
    }
    
    // Change password
    if(isset($_POST['change_password'])) {
        $current_password = $_POST['current_password'];
        $new_password = $_POST['new_password'];
        $confirm_password = $_POST['confirm_password'];
        
        if(empty($current_password) || empty($new_password) || empty($confirm_password)) {
            $error = "All password fields are required";
        } elseif($new_password !== $confirm_password) {
            $error = "New passwords do not match";
        } elseif(strlen($new_password) < 6) {
            $error = "Password must be at least 6 characters";
        } elseif(password_verify($current_password, $user['password'])) {
            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
            $conn->query("UPDATE users SET password = '$hashed_password' WHERE id = $user_id");
            $success = "Password changed successfully!";
        } else {
            $error = "Current password is incorrect";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profile - Project Sale</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/animate.css/4.1.1/animate.min.css">
</head>
<body class="bg-gray-50">
    <!-- Include Header/Sidebar -->
    <?php include 'includes/header.php'; ?>
    
    <div class="flex min-h-screen">
        <!-- Sidebar -->
        <div class="hidden md:block w-64 bg-white shadow-lg">
            <?php include 'includes/sidebar.php'; ?>
        </div>
        
        <!-- Main Content -->
        <div class="flex-1 p-6">
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-3xl font-bold text-gray-800 mb-2">
                    <i class="fas fa-user-circle mr-2"></i> My Profile
                </h1>
                <p class="text-gray-600">Manage your account settings and information</p>
            </div>
            
            <!-- Messages -->
            <?php if($error): ?>
                <div class="bg-red-50 border-l-4 border-red-500 p-4 mb-6 rounded-lg animate__animated animate__shakeX">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-3 text-xl"></i>
                        <span class="text-red-700 font-medium"><?php echo $error; ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <?php if($success): ?>
                <div class="bg-green-50 border-l-4 border-green-500 p-4 mb-6 rounded-lg animate__animated animate__fadeIn">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-3 text-xl"></i>
                        <span class="text-green-700 font-medium"><?php echo $success; ?></span>
                    </div>
                </div>
            <?php endif; ?>
            
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <!-- Left Column - Profile Info -->
                <div class="lg:col-span-2 space-y-8">
                    <!-- Profile Card -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInLeft">
                        <div class="bg-gradient-to-r from-blue-500 to-indigo-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-user-edit mr-2"></i> Profile Information
                            </h3>
                        </div>
                        <div class="p-6">
                            <form method="POST" action="" class="space-y-6">
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    <div>
                                        <label class="block text-gray-700 text-sm font-bold mb-2">
                                            <i class="fas fa-user text-blue-500 mr-2"></i> Username
                                        </label>
                                        <input type="text" value="<?php echo $user['username']; ?>" 
                                               disabled class="w-full px-4 py-3 border border-gray-300 rounded-lg bg-gray-50 text-gray-500">
                                    </div>
                                    
                                    <div>
                                        <label class="block text-gray-700 text-sm font-bold mb-2">
                                            <i class="fas fa-envelope text-blue-500 mr-2"></i> Email
                                        </label>
                                        <input type="email" value="<?php echo $user['email']; ?>" 
                                               disabled class="w-full px-4 py-3 border border-gray-300 rounded-lg bg-gray-50 text-gray-500">
                                    </div>
                                    
                                    <div>
                                        <label class="block text-gray-700 text-sm font-bold mb-2">
                                            <i class="fab fa-telegram text-blue-500 mr-2"></i> Telegram Username
                                        </label>
                                        <input type="text" name="telegram_username" 
                                               value="<?php echo $user['telegram_username'] ?? ''; ?>"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                               placeholder="@username">
                                    </div>
                                    
                                    <div>
                                        <label class="block text-gray-700 text-sm font-bold mb-2">
                                            <i class="fab fa-whatsapp text-green-500 mr-2"></i> WhatsApp Number
                                        </label>
                                        <input type="text" name="whatsapp_number" 
                                               value="<?php echo $user['whatsapp_number'] ?? ''; ?>"
                                               class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition duration-300"
                                               placeholder="+8801XXXXXXXXX">
                                    </div>
                                </div>
                                
                                <div class="pt-4">
                                    <button type="submit" name="update_profile"
                                            class="bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-6 rounded-lg hover:from-blue-700 hover:to-indigo-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 transition duration-300 transform hover:scale-105">
                                        <i class="fas fa-save mr-2"></i> Update Profile
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                    
                    <!-- Account Status Card -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInLeft" style="animation-delay: 0.1s">
                        <div class="bg-gradient-to-r from-purple-500 to-pink-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-shield-alt mr-2"></i> Account Status
                            </h3>
                        </div>
                        <div class="p-6">
                            <div class="space-y-6">
                                <div class="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                                    <div class="flex items-center">
                                        <div class="w-12 h-12 <?php echo $user['is_active'] ? 'bg-green-100' : 'bg-red-100'; ?> rounded-full flex items-center justify-center mr-4">
                                            <i class="fas <?php echo $user['is_active'] ? 'fa-check-circle text-green-600' : 'fa-lock text-red-600'; ?> text-xl"></i>
                                        </div>
                                        <div>
                                            <h4 class="font-bold text-gray-800">Account Status</h4>
                                            <p class="text-gray-600">
                                                <?php echo $user['is_active'] ? 'Your account is fully active' : 'Account requires activation'; ?>
                                            </p>
                                        </div>
                                    </div>
                                    <span class="px-4 py-2 rounded-full font-bold <?php echo $user['is_active'] ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                                        <?php echo $user['is_active'] ? 'ACTIVE' : 'INACTIVE'; ?>
                                    </span>
                                </div>
                                
                                <div class="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                                    <div class="flex items-center">
                                        <div class="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mr-4">
                                            <i class="fas fa-calendar-alt text-blue-600 text-xl"></i>
                                        </div>
                                        <div>
                                            <h4 class="font-bold text-gray-800">Member Since</h4>
                                            <p class="text-gray-600">
                                                <?php echo date('F d, Y', strtotime($user['created_at'])); ?>
                                            </p>
                                        </div>
                                    </div>
                                    <span class="text-gray-600">
                                        <?php 
                                        $days = floor((time() - strtotime($user['created_at'])) / (60 * 60 * 24));
                                        echo $days . ' days';
                                        ?>
                                    </span>
                                </div>
                                
                                <div class="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                                    <div class="flex items-center">
                                        <div class="w-12 h-12 bg-yellow-100 rounded-full flex items-center justify-center mr-4">
                                            <i class="fas fa-key text-yellow-600 text-xl"></i>
                                        </div>
                                        <div>
                                            <h4 class="font-bold text-gray-800">Referral Code</h4>
                                            <p class="text-gray-600">Share this code to earn commissions</p>
                                        </div>
                                    </div>
                                    <div class="flex items-center">
                                        <span class="bg-gray-100 px-4 py-2 rounded-l-lg font-mono font-bold">
                                            <?php echo $user['referral_code']; ?>
                                        </span>
                                        <button onclick="copyText('<?php echo $user['referral_code']; ?>')"
                                                class="bg-blue-600 text-white px-4 py-2 rounded-r-lg hover:bg-blue-700 transition duration-300">
                                            <i class="fas fa-copy"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                            
                            <?php if(!$user['is_active']): ?>
                            <div class="mt-6 pt-6 border-t">
                                <a href="activate.php" 
                                   class="block w-full text-center bg-gradient-to-r from-yellow-400 to-orange-500 text-white font-bold py-3 px-4 rounded-lg hover:opacity-90 transition duration-300 transform hover:scale-105">
                                    <i class="fas fa-bolt mr-2"></i> Activate Account Now
                                </a>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                
                <!-- Right Column - Password & Referral -->
                <div class="space-y-8">
                    <!-- Change Password -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInRight">
                        <div class="bg-gradient-to-r from-green-500 to-emerald-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-lock mr-2"></i> Change Password
                            </h3>
                        </div>
                        <div class="p-6">
                            <form method="POST" action="" class="space-y-4">
                                <div>
                                    <label class="block text-gray-700 text-sm font-bold mb-2">
                                        Current Password
                                    </label>
                                    <input type="password" name="current_password" required
                                           class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 transition duration-300"
                                           placeholder="Enter current password">
                                </div>
                                
                                <div>
                                    <label class="block text-gray-700 text-sm font-bold mb-2">
                                        New Password
                                    </label>
                                    <input type="password" name="new_password" required
                                           class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 transition duration-300"
                                           placeholder="At least 6 characters">
                                </div>
                                
                                <div>
                                    <label class="block text-gray-700 text-sm font-bold mb-2">
                                        Confirm New Password
                                    </label>
                                    <input type="password" name="confirm_password" required
                                           class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 transition duration-300"
                                           placeholder="Re-enter new password">
                                </div>
                                
                                <button type="submit" name="change_password"
                                        class="w-full bg-gradient-to-r from-green-600 to-emerald-700 text-white font-bold py-3 px-4 rounded-lg hover:from-green-700 hover:to-emerald-800 focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2 transition duration-300 mt-4">
                                    <i class="fas fa-key mr-2"></i> Change Password
                                </button>
                            </form>
                        </div>
                    </div>
                    
                    <!-- Referral Stats -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInRight" style="animation-delay: 0.1s">
                        <div class="bg-gradient-to-r from-cyan-500 to-blue-600 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-users mr-2"></i> Referral Stats
                            </h3>
                        </div>
                        <div class="p-6">
                            <?php
                            $ref_stats = $conn->query("
                                SELECT 
                                    COUNT(*) as total,
                                    SUM(CASE WHEN status = 'paid' THEN 1 ELSE 0 END) as active,
                                    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending
                                FROM referrals 
                                WHERE referrer_id = $user_id
                            ")->fetch_assoc();
                            
                            $total_earnings = $conn->query("
                                SELECT SUM(amount) as total FROM transactions 
                                WHERE user_id = $user_id AND type = 'referral_bonus' AND status = 'completed'
                            ")->fetch_assoc()['total'] ?? 0;
                            ?>
                            
                            <div class="space-y-4">
                                <div class="flex justify-between items-center p-3 bg-cyan-50 rounded-lg">
                                    <span class="text-gray-700">Total Referrals</span>
                                    <span class="font-bold text-2xl text-cyan-600"><?php echo $ref_stats['total']; ?></span>
                                </div>
                                
                                <div class="flex justify-between items-center p-3 bg-green-50 rounded-lg">
                                    <span class="text-gray-700">Active Referrals</span>
                                    <span class="font-bold text-2xl text-green-600"><?php echo $ref_stats['active']; ?></span>
                                </div>
                                
                                <div class="flex justify-between items-center p-3 bg-yellow-50 rounded-lg">
                                    <span class="text-gray-700">Pending Referrals</span>
                                    <span class="font-bold text-2xl text-yellow-600"><?php echo $ref_stats['pending']; ?></span>
                                </div>
                                
                                <div class="flex justify-between items-center p-3 bg-purple-50 rounded-lg">
                                    <span class="text-gray-700">Total Earnings</span>
                                    <span class="font-bold text-2xl text-purple-600">৳<?php echo number_format($total_earnings, 2); ?></span>
                                </div>
                            </div>
                            
                            <div class="mt-6">
                                <a href="referrals.php" 
                                   class="block text-center bg-gradient-to-r from-cyan-600 to-blue-700 text-white font-bold py-3 px-4 rounded-lg hover:opacity-90 transition duration-300">
                                    <i class="fas fa-chart-line mr-2"></i> View Referral Details
                                </a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Quick Links -->
                    <div class="bg-white rounded-xl shadow-lg overflow-hidden animate__animated animate__fadeInRight" style="animation-delay: 0.2s">
                        <div class="bg-gradient-to-r from-gray-700 to-gray-900 px-6 py-4">
                            <h3 class="text-xl font-bold text-white">
                                <i class="fas fa-bolt mr-2"></i> Quick Links
                            </h3>
                        </div>
                        <div class="p-6">
                            <div class="space-y-3">
                                <a href="dashboard.php" 
                                   class="flex items-center justify-between p-3 bg-gray-50 hover:bg-blue-50 hover:text-blue-600 rounded-lg transition duration-300 group">
                                    <div class="flex items-center">
                                        <i class="fas fa-tachometer-alt text-gray-400 group-hover:text-blue-500 mr-3"></i>
                                        <span>Dashboard</span>
                                    </div>
                                    <i class="fas fa-chevron-right text-gray-300 group-hover:text-blue-500"></i>
                                </a>
                                
                                <a href="my-orders.php" 
                                   class="flex items-center justify-between p-3 bg-gray-50 hover:bg-green-50 hover:text-green-600 rounded-lg transition duration-300 group">
                                    <div class="flex items-center">
                                        <i class="fas fa-shopping-bag text-gray-400 group-hover:text-green-500 mr-3"></i>
                                        <span>My Orders</span>
                                    </div>
                                    <i class="fas fa-chevron-right text-gray-300 group-hover:text-green-500"></i>
                                </a>
                                
                                <a href="transactions.php" 
                                   class="flex items-center justify-between p-3 bg-gray-50 hover:bg-yellow-50 hover:text-yellow-600 rounded-lg transition duration-300 group">
                                    <div class="flex items-center">
                                        <i class="fas fa-exchange-alt text-gray-400 group-hover:text-yellow-500 mr-3"></i>
                                        <span>Transactions</span>
                                    </div>
                                    <i class="fas fa-chevron-right text-gray-300 group-hover:text-yellow-500"></i>
                                </a>
                                
                                <a href="logout.php" 
                                   class="flex items-center justify-between p-3 bg-gray-50 hover:bg-red-50 hover:text-red-600 rounded-lg transition duration-300 group">
                                    <div class="flex items-center">
                                        <i class="fas fa-sign-out-alt text-gray-400 group-hover:text-red-500 mr-3"></i>
                                        <span>Logout</span>
                                    </div>
                                    <i class="fas fa-chevron-right text-gray-300 group-hover:text-red-500"></i>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script>
    function copyText(text) {
        navigator.clipboard.writeText(text).then(() => {
            // Show notification
            const btn = event.target.closest('button');
            const originalHTML = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-check"></i>';
            btn.classList.remove('bg-blue-600');
            btn.classList.add('bg-green-600');
            
            setTimeout(() => {
                btn.innerHTML = originalHTML;
                btn.classList.remove('bg-green-600');
                btn.classList.add('bg-blue-600');
            }, 2000);
        });
    }
    
    // Add animations
    document.addEventListener('DOMContentLoaded', function() {
        const cards = document.querySelectorAll('.animate__animated');
        cards.forEach((card, index) => {
            setTimeout(() => {
                card.classList.add('animate__animated', 'animate__fadeIn');
            }, index * 100);
        });
    });
    </script>
</body>
</html>